<?php

namespace App\Livewire;

use App\Models\Property;
use App\Models\PropertyImage;
use Filament\Notifications\Notification;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage; 

class PropertyImageGallery extends Component
{
    use WithFileUploads;
    public int $propertyId;
    public $images = [];
    public $imageToDelete = null;
    public $confirmingDelete = false;    
    public array $newImages = [];
    public int $mainImageIndex = 0;
   
   

    public function mount(int $propertyId): void
    {
        $this->propertyId = $propertyId;
        $this->loadImages();
    }

    public function setMainImage(int $index): void
    {
        $this->mainImageIndex = $index;
    }

    public function uploadImages(): void
    {
        $this->validate([
            'newImages.*' => 'image|max:5120',
        ]);

        // Si había una imagen principal previa, la bajamos a level 2
        PropertyImage::where('property_id', $this->propertyId)
            ->where('level', 1)
            ->update(['level' => 2]);

        foreach ($this->newImages as $index => $image) {
            $path = $image->store('properties', 'public');
            $level = $index === $this->mainImageIndex ? 1 : 2;

            PropertyImage::create([
                'property_id' => $this->propertyId,
                'image'       => $path,
                'level'       => $level,
            ]);
        }

        $this->newImages = [];
        $this->mainImageIndex = 0;

        Notification::make()
            ->title('Imágenes guardadas correctamente')
            ->success()
            ->send();


        $this->dispatch('uploadImages');
        $this->loadImages();    
    }

    public function loadImages()
    {
        // ← Esta lógica reemplaza el @php de la vista
        $this->images = PropertyImage::where('property_id', $this->propertyId)
                                        ->orderBy('level')
                                        ->orderBy('id')
                                        ->get();
    }

    public function confirmDelete($imageId)
    {
        $this->imageToDelete = $imageId;
        $this->confirmingDelete = true;
    }

    public function cancelDelete()
    {
        $this->imageToDelete = null;
        $this->confirmingDelete = false;
    }

    public function deleteImage()
    {
        if (!$this->imageToDelete) {
            $this->confirmingDelete = false;
            return;
        }

        $image = PropertyImage::find($this->imageToDelete);

        if ($image && $image->property_id == $this->propertyId) {
            Storage::delete($image->image);
            $image->delete();
            
            Notification::make()
                ->title('Imagen eliminada')
                ->success()
                ->send();
        }

        $this->imageToDelete = null;
        $this->confirmingDelete = false;

        $this->dispatch('imageDeleted');
        $this->loadImages();
    }

    public function setAsPrimary(int $imageId): void
    {
        PropertyImage::where('property_id', $this->propertyId)
            ->update(['level' => 2]);

        PropertyImage::where('id', $imageId)
            ->where('property_id', $this->propertyId)
            ->update(['level' => 1]);

        Notification::make()
            ->title('Imagen principal actualizada')
            ->success()
            ->send();
    }

    public function render()
    {
        return view('livewire.property-image-gallery', [
            //'images' => $this->images,
            'imageToDelete' => $this->imageToDelete,
            'confirmingDelete' => $this->confirmingDelete,
        ]);
    }
}